const ws = new WebSocket("wss://subtitles-robot.com:3333");
let userName = null;
let wsConnectionStartTime;

let showDetails = false;

const infoBlock = document.createElement("div");
infoBlock.style.position = "absolute";
infoBlock.style.top = "0";
infoBlock.style.left = "0";
infoBlock.style.zIndex = "99999998";
infoBlock.style.fontSize = "10px";
infoBlock.style.backgroundColor = "gray";
infoBlock.style.padding = "2px 10px";
infoBlock.style.color = "black";

const infoBlockTitle = document.createElement("span");
infoBlockTitle.textContent = "subtitles robot plugin";
infoBlockTitle.style.fontVariant = "small-caps";
infoBlockTitle.style.fontSize = "12px";
infoBlockTitle.style.paddingRight = "3px";
infoBlockTitle.style.cursor = "pointer";
infoBlockTitle.title = "click to show/hide details";
infoBlockTitle.onclick = () => {
  showDetails = !showDetails;
};

const infoBlockInputLabel = document.createElement("label");
infoBlockInputLabel.textContent = "controller password:";
infoBlockInputLabel.jsFor = "infoBlockInput";

const infoBlockInput = document.createElement("input");
infoBlockInput.style.width = "100px";
infoBlockInput.style.marginLeft = "5px";
infoBlockInput.style.marginRight = "5px";

const infoBlockButton = document.createElement("button");
infoBlockButton.id = "infoBlockButton";
infoBlockButton.innerText = "submit";
infoBlockButton.onclick = () => {
  const password = infoBlockInput.value.trim();
  chrome.storage.local.set({ password: password });
  window.location.reload();
};

const infoBlockStatus = document.createElement("span");
infoBlockStatus.style.marginLeft = "10px";
const infoBlockPosition = document.createElement("span");
infoBlockPosition.style.marginLeft = "10px";
const infoBlockUser = document.createElement("span");
infoBlockUser.style.marginLeft = "10px";
const infoBlockServerHeartbeat = document.createElement("span");
infoBlockServerHeartbeat.style.marginLeft = "10px";
const infoBlockApplicationHeartbeat = document.createElement("span");
infoBlockApplicationHeartbeat.style.marginLeft = "10px";

infoBlock.appendChild(infoBlockTitle);
infoBlock.appendChild(infoBlockInputLabel);
infoBlock.appendChild(infoBlockInput);
infoBlock.appendChild(infoBlockButton);

infoBlock.appendChild(infoBlockStatus);
infoBlock.appendChild(infoBlockPosition);
infoBlock.appendChild(infoBlockUser);
infoBlock.appendChild(infoBlockServerHeartbeat);
infoBlock.appendChild(infoBlockApplicationHeartbeat);

chrome.storage.local.get(["password"]).then((result) => {
  infoBlockInput.value = result.password ? result.password : "";
});

document.body.insertBefore(infoBlock, document.body.firstChild);

ws.onopen = () => {
  chrome.storage.local.get(["password"]).then((result) => {
    if (result.password) {
      ws.send(
        JSON.stringify({
          msgType: "token",
          token: result.password,
          sender: "controller",
        })
      );
    }
  });
};

ws.onmessage = (event) => {
  const message = JSON.parse(event.data);
  if (message.msgType === "tokenOK") {
    userName = message.email;
    wsConnectionStartTime = new Date().getTime();
    sendHeartBeat();
  }
  if (message.msgType === "tokenError") {
    const title = "\nSUBTITLES ROBOT PLUGIN says:\n\n";
    alert(
      message.error === "invalidToken"
        ? `${title}Password '${infoBlockInput.value}' is invalid.\n\nPlease visit https://subtitles-robot.com/controller to get a valid password.`
        : `${title}Password '${infoBlockInput.value}' is used in another controller.`
    );
  } else if (message.msgType === "heartbeat") {
    infoBlockApplicationHeartbeat.textContent =
      "app heartbeat: " +
      Math.abs(new Date().getTime() - parseInt(message.value)) +
      "ms";
  } else if (message.msgType === "controllerHeartbeat") {
    infoBlockServerHeartbeat.textContent =
      "server heartbeat: " +
      Math.abs(new Date().getTime() - parseInt(message.value)) +
      "ms";
  }
};

function sendHeartBeat() {
  setTimeout(() => {
    if (ws && ws.readyState === 1) {
      ws.send(
        JSON.stringify({
          msgType: "heartbeat",
          value: new Date().getTime(),
        })
      );

      sendHeartBeat();
    }
  }, 1000);
}

ws.onerror = (error) => {
  console.log(error);
};
setInterval(() => {
  const video = document.querySelector("video");

  infoBlockUser.style.display = showDetails ? "inline" : "none";
  infoBlockPosition.style.display = showDetails ? "inline" : "none";
  infoBlockServerHeartbeat.style.display = showDetails ? "inline" : "none";
  infoBlockApplicationHeartbeat.style.display = showDetails ? "inline" : "none";

  if (userName) {
    try {
      if (ws && ws.readyState === 1) {
        ws.send(
          JSON.stringify({
            msgType: "position",
            position: video ? video.currentTime : null,
            bookmarkProcessed: true,
          })
        );
      }

      if (userName) {
        infoBlockInputLabel.style.display = "none";
        infoBlockInput.style.display = "none";
        infoBlockButton.style.display = "none";

        const conn =
          ws && ws.readyState === 1
            ? "status: connected"
            : "status: not connected";

        infoBlockPosition.textContent = video
          ? "position: " + formatDurationLong(video.currentTime)
          : "";
        infoBlockUser.textContent = userName ? "user: " + userName : "";

        // const duration = wsConnectionStartTime
        //   ? " duration: " +
        //     formatDurationLong(
        //       Math.floor((new Date().getTime() - wsConnectionStartTime) / 1000)
        //     )
        //   : "";
        if (new Date().getTime() - wsConnectionStartTime > 1_000 * 60 * 4) {
          window.location.reload();
        }
        infoBlockStatus.textContent = " " + conn; // + duration;
        infoBlockStatus.style.color =
          ws && ws.readyState === 1 ? "black" : "darkred";
      } else {
        infoBlockInputLabel.style.display = "inline";
        infoBlockInput.style.display = "inline";
        infoBlockButton.style.display = "inline";
      }
    } catch (error) {
      console.error("position reporter", error);
    }
  }
}, 1000);

function formatDuration(val) {
  let sec_num = parseInt(val, 10);
  let hours = Math.floor(sec_num / 3600);
  let minutes = Math.floor((sec_num - hours * 3600) / 60);
  let seconds = sec_num - hours * 3600 - minutes * 60;
  return [hours, minutes, seconds];
}

function formatDurationLong(val) {
  const tmp = formatDuration(val);
  let hours = tmp[0];
  let minutes = tmp[1];
  let seconds = tmp[2];

  if (hours < 10) {
    hours = "0" + hours;
  }
  if (minutes < 10) {
    minutes = "0" + minutes;
  }
  if (seconds < 10) {
    seconds = "0" + seconds;
  }

  return hours + ":" + minutes + ":" + seconds;
}

function getRoundedLatency(serverHeartBeat) {
  return (
    200 *
    Math.floor(Math.abs(new Date().getTime() - parseInt(serverHeartBeat)) / 200)
  );
}
